/* ***********************************************************************
   *                                                                     *
   * latrig.c                                                            *
   * Christian Zurhorst, BID R&D, 30 Jul. 1998                           *
   *                                                                     *
   * trace memory trigger functions                                      *
   *                                                                     *
   ***********************************************************************

*/

#include <typedefs.h>

#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <latrig.h>
#include <regconst.h>
#include <regx32.h>
#include <regx33.h>
#include <session.h>

extern char perfboard_err[127];


b_errtype EXPORT BestTrigSeqGenPropSet(b_handletype handle,
    b_trigseqgenproptype trigseqgenprop,
    b_int32 value)
{
  b_errtype	err;
  b_int32 cmd_buf[3];           /* array holds command and property value    */
  b_int8 zw[8];
  b_int8ptr bp;
  b_int32 zwl;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    zwl = (b_int32) trigseqgenprop;
    bp = BestLong2Stream(zw, &zwl, 1UL);
    (void) BestLong2Stream(bp, &value, 1UL);
    B_ERRETURN(BestBasicCommand(handle, CMD_TRCTRIG_GENPROPSET, zw,
        IN_TRCTRIG_GENPROPSET, NULL, NULL));
  }
  else
  {
    /* Old 1x protocol */

    /* send command, property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 4UL;           /* this is defined as
                                 * SET_TRIGSEQ_GENERIC_PROP  */
    cmd_buf[1] = trigseqgenprop;/* enumerated property                    */
    cmd_buf[2] = value;

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL);
    /* send 12 bytes */

    /* if onboard error has been detected, receive error message string    */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA, (b_int8ptr) perfboard_err,
        127UL);

      if (err != B_E_OK)
        B_ERRETURN((err));
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }
    else
      B_ERRETURN(err);
  }
}


/*****************************************************************************/
/* This call reads the value of a generic trigger sequencer property         */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqGenPropGet(b_handletype handle,
    b_trigseqgenproptype trigseqgenprop,
    b_int32 * value)
{
  b_errtype err;
  b_int32 cmd_buf[2];
  b_int32 recv_data;
  b_int8 zw[4];
  b_int32 zwl;
  b_int16 outsize;
  b_int8 outbuf[OUT_TRCTRIG_GENPROPGET];
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    /* do the default setting with using the standard call */
    /* add more default settings if needed */

    zwl = (b_int32) trigseqgenprop;
    (void) BestLong2Stream(zw, &zwl, 1UL);
    outsize = OUT_TRCTRIG_GENPROPGET;
    err = BestBasicCommand(handle, CMD_TRCTRIG_GENPROPGET,
      zw, IN_TRCTRIG_GENPROPGET, outbuf, &outsize);
    (void) BestStream2Long(value, outbuf, 1UL);
  }
  else
  {
    /* send command, property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 8UL;           /* this is defined as
                                 * GET_TRIGSEQ_GENERIC_PROP          */
    cmd_buf[1] = trigseqgenprop;/* enumerated property              */

    /* send 8 bytes */
    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL));

    /* receive property value and return it                           */
    B_ERRCHECK(BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL));
    /* read 4 bytes */

    *value = recv_data;

  }
  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call sets all generic trigger sequencer properties to default        */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqGenPropDefaultSet(b_handletype handle)
{
  b_errtype	err;
  b_int32 cmd_type;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    /* do the default setting with using the standard call */
    /* add more default settings if needed */

    /* the default value for the preload counter is 0 */
    B_ERRETURN(BestTrigSeqGenPropSet(handle, B_TRIGSEQGEN_CTRC_PREL, 0UL));

  }
  else
  {
    /* send command,property parameter and data to performance option
     * hardware */

    cmd_type = 0UL;             /* this is defined as
                                 * INIT_ALL_TRIGSEQ_GENERIC_PRO  */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) & cmd_type, 4UL);
    /* send 4 bytes */

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
        B_ERRETURN(err);
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }
    else
      B_ERRETURN(err);
  }
}


/*****************************************************************************/
/*  This call sets an int property of a transient in the trigger sequencer   */
/*  descriptor table                                                         */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqTranPropSet(b_handletype handle,
    b_int32 transient,
    b_trigseqtranproptype trigseqtranprop,
    b_int32 value)
{
  b_errtype	err;
  b_int32 cmd_buf[4];           /* array holds command, transient,property   */
  b_int8 zw[12];
  b_int8ptr bp;
  b_int32 zwl;
  /* and property value                        */

  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);


  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    /* first copy the transient */
    bp = BestLong2Stream(zw, &transient, 1UL);  /* into the bytestream */
    zwl = (b_int32) trigseqtranprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */
    (void) BestLong2Stream(bp, &value, 1UL);  /* and as the last take tye
                                               * value */
    err = BestBasicCommand(handle, CMD_TRCTRIG_TRANPROPSET,
      zw, IN_TRCTRIG_TRANPROPSET, NULL, NULL);
  }
  else
  {
    /* send command,property parameter and data to performance option
     * hardware */

    cmd_buf[0] = 2UL;           /* this is defined as
                                 * SET_TRIGSEQ_TRANSIENT_PROP   */
    cmd_buf[1] = transient;
    cmd_buf[2] = trigseqtranprop; /* enumerated property                     */
    cmd_buf[3] = value;         /* property value                          */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 16UL);
    /* send 16 bytes */

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
        B_ERRETURN(err);
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }
  }

  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call reads the value of an int based property from the property      */
/* register set                                                              */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqTranPropGet(b_handletype handle,
    b_int32 transient,
    b_trigseqtranproptype trigseqtranprop,
    b_int32 * value)
{
  b_errtype	err;
  b_int32 cmd_buf[3];
  b_int32 recv_data;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!BestIs2925(handle))
  {
    B_ERRETURN(B_E_ONLY_E2925A);
  }

  /* send command, property parameter and data to performance option hardware */

  cmd_buf[0] = 7UL;             /* this is defined as
                                 * GET_TRIGSEQ_TRANSIENT_PROP */
  cmd_buf[1] = transient,
    cmd_buf[2] = trigseqtranprop; /* enumerated property                     */

  B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL));
  /* 12 bytes */

  /* receive property value and return it                                    */

  B_ERRCHECK(BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL));
  /* 4 bytes */

  *value = recv_data;

  B_ERRETURN(err);

}


/*****************************************************************************/
/*  This call initialises a transient in the Trigger Sequencer Descriptor    */
/*  Table                                                                    */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqTranPropDefaultSet(b_handletype handle,
    b_int32 transient)
{
  b_errtype	err;
  b_int32 cmd_buf[2];           /* performance option command */
  b_int8 zw[4];
  /* first check if handle is good                                           */


  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    (void) BestLong2Stream(zw, &transient, 1UL);  /* into the bytestream */
    err = BestBasicCommand(handle, CMD_TRCTRIG_TRANPROPDEFAULTSET,
      zw, IN_TRCTRIG_TRANPROPDEFAULTSET, NULL, NULL);

    /* make sure that the default for xcond is always '1' */
    if (err == B_E_OK)
      err = BestTrigSeqTranCondPropSet(handle, transient,
        B_TRIGSEQ_XCOND, "1");
  }
  else
  {
    /* send command to performance option hardware                         */

    cmd_buf[0] = 3UL;           /* this is INIT_ALL_TRIGSEQ_TRANSIENT_PROP */
    cmd_buf[1] = transient;

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 8UL);

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
        B_ERRETURN(err);
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }

  }
  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call sets a string property of a transient in the trigger sequencer  */
/* descriptor table                                                          */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqTranCondPropSet(b_handletype handle,
    b_int32 transient,
    b_trigseqtrancondproptype trigseqtrancondprop,
    b_charptrtype condition)
{
  B_DECLARE_FUNCNAME("BestTrigSeqTranCondPropSet [tstrancprpset]");

  b_errtype	err;
  char sync = '\1';             /* start of string marker                    */
  b_int32 cmd_buf[3];           /* command buffer                            */
  b_int8ptr zw;                 /* the buffer for the string */
  b_int8ptr bp;
  b_int32 zwl;
  b_int32 length;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);
  B_FCT_PARAM_NULL_POINTER_CHK(condition);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */

    length = 9UL + (b_int32) strlen(condition);
    zw = (b_int8ptr) malloc((size_t) length);
    /* first copy the transient */
    bp = BestLong2Stream(zw, &transient, 1UL);  /* into the bytestream */
    zwl = (b_int32) trigseqtrancondprop;
    bp = BestLong2Stream(bp, &zwl, 1UL);  /* then take the property */
    (void) BestByteCopy(bp, (b_int8ptr) condition, strlen(condition) + 1UL);  /* and last is condition */

    err = BestBasicCommandVariable(handle, CMD_TRCTRIG_TRANCONDPROPSET,
      zw, (b_int16) length, NULL, NULL);

    free(zw);                   /* free the allocated memory */
  }
  else
  {
    cmd_buf[0] = 2UL;
    cmd_buf[1] = transient;
    cmd_buf[2] = trigseqtrancondprop + B_TRIGSEQ_CONDITION_OFFSET;

    /* first transfer condition string via the performance data port        */

    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) & sync, 1UL));
    B_ERRCHECK(BestBasicBlockWrite(handle, PERF_DATA, (b_int8ptr) condition,
        (b_int32) strlen(condition) + 1UL));

    /* then send command, transient and property parameter to performance
     * hardware */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL);

    /* if onboard error has been detected, receive error message string    */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
        B_ERRETURN(err);
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }
  }
  B_ERRETURN(err);

}


/*****************************************************************************/
/*This call reads a string property from the property register set. Note that*/
/* this call returns the +parsed+ string property, that is, a 32bit function */
/* map                                                                       */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqTranCondPropGet(b_handletype handle,
    b_int32 transient,
    b_trigseqtrancondproptype trigseqtrancondprop,
    b_int32 * value)
{
  b_errtype	err;
  b_int32 cmd_buf[3];
  b_int32 recv_data;
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (!BestIs2925(handle))
  {
    B_ERRETURN(B_E_ONLY_E2925A);
  }

  /* send command, property parameter and data to performance option hardware */

  cmd_buf[0] = 7UL;             /* this is defined as GET_SEQ_PROP          */
  cmd_buf[1] = transient;
  cmd_buf[2] = trigseqtrancondprop + B_TRIGSEQ_CONDITION_OFFSET;  /* enumerated property                */

  B_ERRCHECK(BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) cmd_buf, 12UL));
  /* send 12 bytes */

  /* receive property value and return it                                    */

  B_ERRCHECK(BestBasicBlockRead(handle, PERF_COM, (b_int8ptr) & recv_data, 4UL));
  /* read 4 bytes */

  *value = recv_data;

  B_ERRETURN(B_E_OK);

}


/*****************************************************************************/
/*  This call initialises the Trigger Sequencer Descriptor Table             */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqPropDefaultSet(b_handletype handle)
{
  b_errtype	err;
  b_int32 cmd_type;             /* performance option command */
  /* first check if handle is good                                           */

  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_TRCTRIG_PROPDEFAULTSET,
      NULL, IN_TRCTRIG_PROPDEFAULTSET, NULL, NULL);

  }
  else
  {
    /* send command to performance option hardware                       */

    cmd_type = 1UL;             /* this is CLEAR_SEQ_DESCR_TABLE */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) & cmd_type, 4UL);

    /* if onboard error has been detected, receive error message string     */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
        B_ERRETURN(err);
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }

  }
  B_ERRETURN(err);
}


/*****************************************************************************/
/* This call programs the trigger sequencer memory                           */
/*****************************************************************************/

b_errtype EXPORT BestTrigSeqProg(b_handletype handle)
{
  b_errtype	err;
  b_int32 cmd_type;
  B_LICENSECHECK(B_CAPABILITY_ANALYZER);

  if (Best16BitRegisterFile(handle))
  {
    /* New 2x protocol */
    err = BestBasicCommand(handle, CMD_TRC_TRIGPROG,
      NULL, IN_TRC_TRIGPROG, NULL, NULL);
  }
  else
  {
    /* send command to performance option hardware                          */

    cmd_type = 5UL;             /* this is PROG_TRIGSEQ_MEMORY   */

    err = BestBasicBlockWrite(handle, PERF_COM, (b_int8ptr) & cmd_type, 4UL);

    /* if onboard error has been detected, receive error message string      */

    if (err != B_E_OK)
    {
      err = BestBasicBlockRead(handle, PERF_DATA,
        (b_int8ptr) perfboard_err, 127UL);

      if (err != B_E_OK)
        B_ERRETURN(err);
      else
        B_ERRETURN(B_E_PERFBOARD_ERROR);
    }
  }

  B_ERRETURN(err);
}


